<?php
/**
 * Plugin Name: Defform Contact Form
 * Plugin URI: https://defform.com/
 * Description: Simple contact form plugin with AI-powered spam and sales email filtering.
 * Version: 1.0.0
 * Author: ToDoONada
 * Author URI: https://todoonada.co.jp/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: defform
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

/*  Copyright 2026 ToDoONada.inc

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 2, as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

if (!defined('ABSPATH')) {
    exit;
}

// 定数
define('DEFFORM_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('DEFFORM_PLUGIN_URL', plugin_dir_url(__FILE__));

// クラス読み込み
require_once DEFFORM_PLUGIN_PATH . 'includes/class-form-manager.php';

// プラグイン初期化
add_action('init', 'DefformDashboard::init');

class DefformDashboard
{
    const PLUGIN_ID = 'defform-dashboard';

    private static $instance = null;
    private $form_manager;

    static function init()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    function __construct()
    {
        $this->form_manager = new DefformFormManager();

        if (is_admin() && is_user_logged_in()) {
            add_action('admin_menu', [$this, 'add_plugin_menu']);
            add_action('admin_init', [$this->form_manager, 'handle_form_submit']);
            add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);
        }

        // ショートコード登録（管理画面以外でも動くように外に出す）
        add_shortcode('defform', [$this, 'render_shortcode']);
    }

    // メニュー追加
    function add_plugin_menu()
    {
        add_menu_page(
            'Defform Dashboard',
            'Defform',
            'manage_options',
            self::PLUGIN_ID,
            [$this, 'show_top_page'],
            'dashicons-cloud',
            30
        );

        add_submenu_page(
            self::PLUGIN_ID,
            'Defform Dashboard',
            'トップ',
            'manage_options',
            self::PLUGIN_ID,
            [$this, 'show_top_page']
        );

        add_submenu_page(
            self::PLUGIN_ID,
            'フォーム管理',
            'フォーム管理',
            'manage_options',
            self::PLUGIN_ID . '-forms',
            [$this, 'show_forms_page']
        );
    }

    // CSS・JS読み込み
    function enqueue_styles($hook)
    {
        if (strpos($hook, 'defform') !== false) {
            wp_enqueue_style(
                'defform-admin-css',
                DEFFORM_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                '1.0.0'
            );
            wp_enqueue_script(
                'defform-admin-copy',
                DEFFORM_PLUGIN_URL . 'assets/js/admin-copy.js',
                array(),
                '1.0.0',
                true
            );
        }
    }

     // ショートコード処理
    function render_shortcode($atts)
    {
        $atts = shortcode_atts(array(
            'id' => 0,
            'container' => '',
        ), $atts);

        $id = intval($atts['id']);

        if ($id === 0) {
            return '<!-- Defform: IDが指定されていません -->';
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'defform_forms';
        $form = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d AND is_active = 1",
            $id
        ));

        if (!$form) {
            return '<!-- Defform: フォームが見つかりません -->';
        }

        if (empty($form->js_file_path)) {
            return '<!-- Defform: JSファイルが設定されていません -->';
        }

        // defform_idがあればそれを使用、なければフォールバック
        $container_id = !empty($form->defform_id)
            ? sanitize_html_class($form->defform_id)
            : 'defform-' . $id;

        $handle = 'defform-form-' . $id;

        wp_enqueue_script(
            $handle,
            $form->js_file_path,
            array(),
            '1.0.0',
            true
        );

        // script タグに data-target 属性を追加
        add_filter('script_loader_tag', function ($tag, $registered_handle) use ($handle, $container_id) {
            if ($registered_handle === $handle) {
                $tag = str_replace(
                    ' src=',
                    ' data-target="' . esc_attr($container_id) . '" src=',
                    $tag
                );
            }
            return $tag;
        }, 10, 2);

        $html  = '<div id="' . esc_attr($container_id) . '"></div>' . "\n";

        return $html;
    }

    // トップページ表示
    function show_top_page()
    {
        include DEFFORM_PLUGIN_PATH . 'views/top.php';
    }

    // フォーム管理ページ表示
    function show_forms_page()
    {
        $action = isset($_GET['action']) ? $_GET['action'] : 'list';

        switch ($action) {
            case 'add':
                include DEFFORM_PLUGIN_PATH . 'views/form-add.php';
                break;
            case 'edit':
                include DEFFORM_PLUGIN_PATH . 'views/form-edit.php';
                break;
            default:
                $forms = $this->form_manager->get_all_forms();
                include DEFFORM_PLUGIN_PATH . 'views/forms.php';
                break;
        }
    }
}

// ============================================
// テーブル作成
// ============================================
function defform_create_table()
{
    global $wpdb;

    $table_name = $wpdb->prefix . 'defform_forms';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        form_name varchar(255) NOT NULL,
        defform_id varchar(255) DEFAULT '',
        form_url varchar(500) DEFAULT '',
        js_file_path varchar(500) DEFAULT '',
        is_active tinyint(1) DEFAULT 1,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

register_activation_hook(__FILE__, 'defform_create_table');