<?php
if (!defined('ABSPATH')) {
    exit;
}

class DefformFormManager
{
    private $table_name;

    public function __construct()
    {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'defform_forms';
    }

    // 全フォーム取得
    public function get_all_forms()
    {
        global $wpdb;
        return $wpdb->get_results(
            "SELECT * FROM {$this->table_name} ORDER BY id ASC"
        );
    }

    // 1件取得
    public function get_form($id)
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE id = %d",
            intval($id)
        ));
    }

    // フォーム送信処理
    public function handle_form_submit()
    {
        if (!isset($_POST['defform_action'])) {
            return;
        }

        if (!check_admin_referer('defform_form_action', 'defform_nonce')) {
            return;
        }

        $action = $_POST['defform_action'];

        switch ($action) {
            case 'add_form':
                $this->add_form();
                break;
            case 'edit_form':
                $this->edit_form();
                break;
            case 'delete_form':
                $this->delete_form();
                break;
            case 'toggle_form':
                $this->toggle_form();
                break;
        }
    }

    // フォーム追加
    private function add_form()
    {
        global $wpdb;

        $form_name  = sanitize_text_field($_POST['form_name'] ?? '');
        $defform_id = sanitize_text_field($_POST['defform_id'] ?? '');
        $form_url   = esc_url_raw($_POST['form_url'] ?? '');

        if (empty($form_name)) {
            set_transient('defform_message', 'フォーム名を入力してください', 30);
            set_transient('defform_message_type', 'error', 30);
            return;
        }

        $js_file_path = '';
        if (!empty($_FILES['js_file']['name'])) {
            $js_file_path = $this->upload_js_file($_FILES['js_file']);

            if (is_wp_error($js_file_path)) {
                set_transient('defform_message', $js_file_path->get_error_message(), 30);
                set_transient('defform_message_type', 'error', 30);
                return;
            }
        }

        $result = $wpdb->insert(
            $this->table_name,
            array(
                'form_name'    => $form_name,
                'defform_id'   => $defform_id,
                'form_url'     => $form_url,
                'js_file_path' => $js_file_path,
                'is_active'    => 1,
            ),
            array('%s', '%s', '%s', '%s', '%d')
        );

        if ($result === false) {
            set_transient('defform_message', '保存に失敗しました', 30);
            set_transient('defform_message_type', 'error', 30);
        } else {
            set_transient('defform_message', 'フォームを追加しました！', 30);
            set_transient('defform_message_type', 'success', 30);
        }

        wp_safe_redirect(admin_url('admin.php?page=defform-dashboard-forms'));
        exit;
    }

    // フォーム編集
    private function edit_form()
    {
        global $wpdb;

        $id         = intval($_POST['form_id'] ?? 0);
        $form_name  = sanitize_text_field($_POST['form_name'] ?? '');
        $defform_id = sanitize_text_field($_POST['defform_id'] ?? '');
        $form_url   = esc_url_raw($_POST['form_url'] ?? '');

        if ($id === 0 || empty($form_name)) {
            set_transient('defform_message', 'フォーム名を入力してください', 30);
            set_transient('defform_message_type', 'error', 30);
            return;
        }

        // 更新データ
        $update_data = array(
            'form_name'  => $form_name,
            'defform_id' => $defform_id,
            'form_url'   => $form_url,
        );
        $update_format = array('%s', '%s', '%s');

        // 新しいJSファイルがアップロードされた場合
        if (!empty($_FILES['js_file']['name'])) {
            $js_file_path = $this->upload_js_file($_FILES['js_file']);

            if (is_wp_error($js_file_path)) {
                set_transient('defform_message', $js_file_path->get_error_message(), 30);
                set_transient('defform_message_type', 'error', 30);
                return;
            }

            // 古いファイルを削除
            $old_form = $this->get_form($id);
            if ($old_form && !empty($old_form->js_file_path)) {
                $this->delete_js_file($old_form->js_file_path);
            }

            $update_data['js_file_path'] = $js_file_path;
            $update_format[] = '%s';
        }

        $result = $wpdb->update(
            $this->table_name,
            $update_data,
            array('id' => $id),
            $update_format,
            array('%d')
        );

        if ($result === false) {
            set_transient('defform_message', '更新に失敗しました', 30);
            set_transient('defform_message_type', 'error', 30);
        } else {
            set_transient('defform_message', 'フォームを更新しました！', 30);
            set_transient('defform_message_type', 'success', 30);
        }

        wp_safe_redirect(admin_url('admin.php?page=defform-dashboard-forms'));
        exit;
    }

    // フォーム削除
    private function delete_form()
    {
        global $wpdb;

        $id = intval($_POST['form_id'] ?? 0);

        if ($id === 0) {
            return;
        }

        // JSファイルも削除
        $form = $this->get_form($id);
        if ($form && !empty($form->js_file_path)) {
            $this->delete_js_file($form->js_file_path);
        }

        $result = $wpdb->delete(
            $this->table_name,
            array('id' => $id),
            array('%d')
        );

        if ($result === false) {
            set_transient('defform_message', '削除に失敗しました', 30);
            set_transient('defform_message_type', 'error', 30);
        } else {
            set_transient('defform_message', 'フォームを削除しました', 30);
            set_transient('defform_message_type', 'success', 30);
        }

        wp_safe_redirect(admin_url('admin.php?page=defform-dashboard-forms'));
        exit;
    }

    // 有効/無効切り替え
    private function toggle_form()
    {
        global $wpdb;

        $id = intval($_POST['form_id'] ?? 0);

        if ($id === 0) {
            return;
        }

        $form = $this->get_form($id);
        if (!$form) {
            return;
        }

        // 有効↔無効を切り替え
        $new_status = $form->is_active ? 0 : 1;

        $result = $wpdb->update(
            $this->table_name,
            array('is_active' => $new_status),
            array('id' => $id),
            array('%d'),
            array('%d')
        );

        $status_text = $new_status ? '有効' : '無効';

        if ($result === false) {
            set_transient('defform_message', 'ステータスの変更に失敗しました', 30);
            set_transient('defform_message_type', 'error', 30);
        } else {
            set_transient('defform_message', "フォームを{$status_text}にしました", 30);
            set_transient('defform_message_type', 'success', 30);
        }

        wp_safe_redirect(admin_url('admin.php?page=defform-dashboard-forms'));
        exit;
    }

    // JSファイルアップロード
    private function upload_js_file($file)
    {
        // 拡張子チェック
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($file_ext !== 'js') {
            return new WP_Error('invalid_type', 'JSファイル（.js）のみアップロード可能です');
        }

        // MIMEタイプチェック
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);

        $allowed_mimes = array(
            'application/javascript',
            'text/javascript',
            'application/x-javascript',
            'text/plain',
        );

        if (!in_array($mime, $allowed_mimes, true)) {
            return new WP_Error('invalid_type', '不正なファイルタイプです');
        }

        $upload_overrides = array(
            'test_form' => false,
            'mimes'     => array(
                'js' => 'application/javascript',
            ),
        );

        // WordPress標準のアップロード処理を使用
        $uploaded = wp_handle_upload($file, $upload_overrides);

        if (isset($uploaded['error'])) {
            return new WP_Error('upload_failed', $uploaded['error']);
        }

        return $uploaded['url'];
    }

    // JSファイル削除
    private function delete_js_file($file_url)
    {
        $upload_dir = wp_upload_dir();
        $file_path = str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $file_url);

        if (file_exists($file_path)) {
            unlink($file_path);
        }
    }
}